# R Code for Simulation of Harmonized Compendial Dissolution Criteria USP <711>
# For Immediate Release Products
# January 2017
# Lori B. Pfahler (lori.pfahler@merck.com)
# 
# Programmed using the following R Version:
# R version 3.3.2 (2016-10-31) -- "Sincere Pumpkin Patch"
# Copyright (C) 2016 The R Foundation for Statistical Computing
# Platform: x86_64-w64-mingw32/x64 (64-bit)


# Simulation function for USP dissolution - three stage test
# reps =   number of simulated lots for simulation
# mean =   the process mean
# sd =     the process standard deviation
# Q =      value as specified in the mongraph for the product
#
# Summary Matrix Key
# ==================
# 1 "PassS1"
# 2 "PassS2" 
# 3 "PassS3"
# 4 "#<Q S1"
# 5 "<Q-25 S1"
# 6 "Avg12"
# 7 "#<Q-15 S2"
# 8 "#<Q-25 S2"
# 9 "Avg24"
# 10 "#<Q-15 S3"
# 11 "#<Q-25 S3"
# 12 "PassAll"
# 13 "GoS2"
# 14 "GoS3"


simDissoTest <- function(reps=100, mean=100, sd=5, Q=80)
{
  # generate data following user chosen mean and sd
  # generate 24 data values even if not needed (that is
  # not going to all three stages)
  
  data.matrix <- matrix(rnorm(24*reps,mean=mean, sd=sd), nrow=reps, ncol=24)	
  
  # Summary Matrix to hold simulation results
  
  sum.matrix <- matrix(data=NA, nrow=reps, ncol=14, 
    dimnames=list(NULL, c("PassS1", "PassS2", "PassS3", "nLTQ S1", 
    "nLTQ-25 S1", "Avg12", "nLTQ-15 S2", "nLTQ-25 S2", "Avg24", 
    "nLTQ-15 S3", "nLTQ-25 S3", "PassAll", "GoS2", "GoS3")))
  
  # Calculate summary values needed
  sum.matrix[,4] <- rowSums(data.matrix[,1:6] < Q+5)
  sum.matrix[,5] <- rowSums(data.matrix[,1:6] < Q-25)	
  sum.matrix[,6] <- rowMeans(data.matrix[,1:12])
  sum.matrix[,7] <- rowSums(data.matrix[,1:12] < Q-15)
  sum.matrix[,8] <- rowSums(data.matrix[,1:12] < Q-25)	
  sum.matrix[,9] <- rowMeans(data.matrix[,1:24])
  sum.matrix[,10] <- rowSums(data.matrix[,1:24] < Q-15)	
  sum.matrix[,11] <- rowSums(data.matrix[,1:24] < Q-25)	
  
  for(i in 1:reps) 
  {	
    # Stage 1
    if(sum.matrix[i,4]==0)
    {
      sum.matrix[i,1] <- 1
      sum.matrix[i,12] <- 1			
      sum.matrix[i,13] <- 0
      sum.matrix[i,14] <- 0
    }
    else
    {
      sum.matrix[i,1] <- 0
      # Check to see if any results are below Q-25 Stage 1
      # If there are any - do not proceed - fail test
      if(sum.matrix[i,5]==0)
      {
        sum.matrix[i,13] <- 1
      }
      else
      {
        sum.matrix[i,12] <- 0
        sum.matrix[i,13] <- 0
        sum.matrix[i,14] <- 0
      }
    }
    
    # Stage 2
    if(sum.matrix[i,13]==1)
    {
      # Check to see if any results are below Q-25 in Stage 2
      # and there are more than 2 units that are less than Q-15
      # If there are any - do not proceed - fail test
      if(sum.matrix[i,6]>=Q & sum.matrix[i,7]==0)
      {
        sum.matrix[i,2] <- 1
        sum.matrix[i,12] <- 1			
        sum.matrix[i,14] <- 0
      }
      else
      {
        sum.matrix[i,2] <- 0
        if(sum.matrix[i,7]<=2 & sum.matrix[i,8]==0)
        {
          sum.matrix[i,14] <- 1
        }
        else
        {
          sum.matrix[i,14] <- 0
          sum.matrix[i,12] <- 0
        }
      }
    }
    
    # Stage 3
    if(sum.matrix[i,14]==1)
    {
      if(sum.matrix[i,9]>=Q & sum.matrix[i,10]<=2 & sum.matrix[i,11]==0)
      {
        sum.matrix[i,3] <- 1
        sum.matrix[i,12] <- 1			
      }
      else
      {
        sum.matrix[i,3] <- 0
        sum.matrix[i,12] <- 0			
      }
    }
  }
  return(data.frame(sum.matrix))
}


# simRun1 is a test run only to test the function
# simulation run with 10,000 batches with the mean = 82% and 
# sd = 4% and the Q value set to 80% dissolved/%LC
simRun1 <- simDissoTest(reps=10000, mean=82, sd=4, Q=80)
# percent pass Stage 1
mean(simRun1[,1])*100
# percent pass at Stage 1 or 2
mean(simRun1[,1]==1 | simRun1[,2]==1)*100
# percent pass overall
mean(simRun1[,12])*100





                                    
                                    












